/*
Generic Unix sub-include file for crtc.c

This file implements some of the functions, as described in crtdummy.h,
for a generic Unix system. It is meant to be included into other include
files for more specific Unix systems.

Catching the interrupt signal as well as refreshing the window is
implemented; Sound, NoSound and GetShiftState are not. Changing the
size of the screen after receiving a WINCH signal is implemented,
but TextMode cannot generate this signal.

Copyright (C) 1998-99 Free Software Foundation, Inc.

Author: Frank Heckenbach <frank@pascal.gnu.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation, version 2.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; see the file COPYING.LIB.
If not, write to the Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

As a special exception, if you link this library with files compiled
with a GNU compiler to produce an executable, this does not cause
the resulting executable to be covered by the GNU Library General
Public License. This exception does not however invalidate any other
reasons why the executable file might be covered by the GNU Library
General Public License.
*/

#include <sys/time.h>

/* Time for delayed refreshes */
#define REFRESH_SECONDS 0
#define REFRESH_MICROSECONDS 100000

static void (*old_winch_handler) (int);

static void alarm_handler (int sig)
{
  (void) sig;
  crt_refresh ();
}

static void crt_schedule_refresh ()
{
  static struct itimerval timerval = { { 0, 0 }, { REFRESH_SECONDS, REFRESH_MICROSECONDS } };
  signal (SIGALRM, &alarm_handler);
  setitimer (ITIMER_REAL, &timerval, 0);
}

static void crt_stop_refresh ()
{
  static struct itimerval notimerval = { { 0, 0 }, { 0, 0 } };
  signal (SIGALRM, &alarm_handler);
  setitimer (ITIMER_REAL, &notimerval, 0);
}

static void int_handler (int sig)
{
  (void) sig;
  crt_signal_handler (0x100 * ksInt);
  signal (SIGINT, &int_handler);
}

static void term_handler (int sig)
{
  (void) sig;
  crt_signal_handler (0x100 * ksTerm);
  signal (SIGTERM, &term_handler);
}

static void hup_handler (int sig)
{
  (void) sig;
  crt_signal_handler (0x100 * ksHUp);
  signal (SIGHUP, &hup_handler);
}

static void winch_handler (int sig)
{
  crt_screen_size_changed++;
  if (old_winch_handler != SIG_ERR && old_winch_handler != SIG_IGN && old_winch_handler != SIG_DFL)
    (*old_winch_handler) (sig);
  signal (SIGWINCH, &winch_handler);
}

static void crt_special_init1 ()
{
  signal (SIGINT,  &int_handler);
  signal (SIGTERM, &term_handler);
  signal (SIGHUP,  &hup_handler);
  old_winch_handler = signal (SIGWINCH, &winch_handler);
}
