/*
Support routines for the CRT unit

Copyright (C) 1998-99 Free Software Foundation, Inc.

Author: Frank Heckenbach <frank@pascal.gnu.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation, version 2.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; see the file COPYING.LIB.
If not, write to the Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

As a special exception, if you link this library with files compiled
with a GNU compiler to produce an executable, this does not cause
the resulting executable to be covered by the GNU Library General
Public License. This exception does not however invalidate any other
reasons why the executable file might be covered by the GNU Library
General Public License.
*/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <signal.h>
#include "crtc.h"

int crt_get_input_fd (void);
int crt_get_output_fd (void);

#ifdef USE_PDCURSES
#ifdef XCURSES
#include <xcurses.h>
#include <xpanel.h>
char *XCursesProgramName;
#else
#include <curses.h>
#include <panel.h>
#endif
#define REVERSE_WORKS FALSE
#define INVIS_WORKS   TRUE
#define HAVE_RAW_OUTPUT
#define FKEYSH   13
#define FKEYCTRL 25
#define FKEYALT  37

/* This is undocumented, but there doesn't seem to be another way! */
#define SETCURSOR(x,y) wmove (curscr, y, x)

#ifdef XCURSES
static void clrscr ()
{
}
#else
/* NOTE: works only for DJGPP -- other PDCurses targets (e.g. win32)
   probably don't need to clear the screen and can be handled in the
   ifdef above. */
/* #include <conio.h> -- conflicts with curses */ extern void clrscr (void);
#endif

int crt_get_input_fd ()
{
  return PDC_get_input_fd ();
}

int crt_get_output_fd ()
{
#ifdef XCURSES
  return - 1;
#else
  return fileno (stdout);
#endif
}
#endif

#ifdef USE_NCURSES
#include <ncurses.h>
#include <panel.h>
#include <term.h>
#define REVERSE_WORKS TRUE
#define INVIS_WORKS   FALSE
#undef  HAVE_RAW_OUTPUT
#define FKEYSH   11
#define FKEYCTRL 21
#define FKEYALT  31

/* This is undocumented, but there doesn't seem to be another way! */
#define SETCURSOR(x,y) wmove (newscr, y, x)

static int putch (int c)
{
  return putchar (c);
}

static void clrscr ()
{
  tputs (clear_screen, lines > 0 ? lines : 1, putch);
  fflush (stdout);
}

int crt_get_input_fd ()
{
  return fileno (stdin);
}

int crt_get_output_fd ()
{
  return fileno (stdout);
}
#endif

#ifndef A_ALTCHARSET
#define A_ALTCHARSET 0
#endif

typedef unsigned char Boolean;
typedef unsigned int  TKey;

int     crt_VirtualShiftState = 0, crt_screen_size_changed = 0;
Boolean crt_Signaled = FALSE, crt_CheckBreak = TRUE;

extern char *crt_get_program_name (void);
extern char **crt_get_c_environment (void);
extern void crt_signal_handler (TKey sig);
extern void crt_fatal (int);

static void crt_refresh (void);

int crt_getshiftstate (void);
void crt_sound (unsigned Hz);
void crt_nosound (void);

#ifdef XCURSES
#include "crtxcurses.h"
#elif defined(__linux__) && defined(__i386__)
#include "crtlinux386.h"
#elif defined(MSDOS) && defined(USE_PDCURSES)
#include "crtdospc.h"
#elif defined(__unix__)
#include "crtunix.h"
#else
#include "crtdummy.h"
#endif

#define MAXLENGTH 4096

typedef unsigned char Char;
typedef unsigned char TTextAttr;
typedef enum { CursorIgnored, CursorHidden, CursorNormal, CursorFat, CursorBlock } TCursorShape;
typedef enum { UpdateNever, UpdateWaitInput, UpdateInput, UpdateRegularly, UpdateAlways } TCRTUpdate;
typedef struct { int x, y; } TPoint;
typedef struct { Char Ch; TTextAttr Attr; } TCharAttr;
typedef struct
{
  int curses_key;
  Char crt_key;
} TKeyTable;
typedef struct TPanelData
{
  struct TPanelData *Next;
  WINDOW      *w;
  PANEL       *panel;
  Boolean      BoundToBackground, Hidden, ScrollState, PCCharSet, UseControlChars;
  TCursorShape CursorShape;
  unsigned int WindMin, WindMax, LastWindMin, LastWindMax;
  TTextAttr    TextAttr;
} TPanelData;
typedef TPanelData* TPanel;

TCRTUpdate      crt_update_level = UpdateNever;
TCursorShape    crt_last_shape = - 1;
TPoint          crt_ScreenSize;
TTextAttr       crt_TextAttr, crt_NormAttr;
TKey            crt_unget_key_buf = 0;
Char            crt_fkey_buf = 0, crt_line_buf [MAXLENGTH], *crt_line_buf_pos;
size_t          crt_line_buf_count = 0;
chtype          crt_attrs [8] [8], crt_monoattrs [8] [8];
void          (*crt_old_tstp_handler) (int) = NULL;
unsigned int    crt_LastMode, crt_WindMin, crt_WindMax;
TPanel          crt_ActivePanel = NULL, crt_PanelList = NULL;
int             crt_inited = 0, crt_key_buf = 0,
                crt_refresh_inhibit = 0, crt_refresh_flag = 0,
                crt_colors [8] = { COLOR_BLACK, COLOR_BLUE,    COLOR_GREEN,  COLOR_CYAN,
                                   COLOR_RED,   COLOR_MAGENTA, COLOR_YELLOW, COLOR_WHITE };
Boolean         crt_ColorFlag, crt_HasColors, crt_linux_console = 0,
                crt_LastCheckBreak = - 1, crt_CheckEOF = FALSE, crt_VisualBell = FALSE,
                crt_terminal_no_crt = FALSE, crt_pending_refresh = FALSE,
                crt_XCRT =
                           #ifdef XCURSES
                           TRUE;
                           #else
                           FALSE;
                           #endif

#define WindXMin (crt_WindMin & 0xff)
#define WindXMax (crt_WindMax & 0xff)
#define WindYMin (crt_WindMin >> 8)
#define WindYMax (crt_WindMax >> 8)

#ifndef KEY_RESIZE
#define KEY_RESIZE (KEY_MAX + 1)
#endif

static TKeyTable KeyTable [] =
{
  { KEY_BACKSPACE, chBkSp },
  { KEY_SUSPEND,   26     },
  { '\n',          chCR   },
  #ifdef USE_PDCURSES
  { '\r',          chCR   },
  { PADSTAR,       '*'    },
  { PADMINUS,      '-'    },
  { PADPLUS,       '+'    },
  { CTL_ENTER,     chLF   },
  { CTL_PADSTAR,   chLF   },
  { CTL_PADMINUS,  chCR   },
  { CTL_PADPLUS,   11     },
  #else
  { '\r',          chLF   },
  #endif
  { 0,             0      }
};

static TKeyTable FKeyTable [] =
{
  { KEY_RESIZE,    ksScreenSizeChanged },
  { KEY_BTAB,      ksShTab     },
  { KEY_LEFT,      ksLeft      },
  { KEY_RIGHT,     ksRight     },
  { KEY_UP,        ksUp        },
  { KEY_DOWN,      ksDown      },
  { KEY_A3,        ksPgUp      },
  { KEY_PPAGE,     ksPgUp      },
  { KEY_C3,        ksPgDn      },
  { KEY_NPAGE,     ksPgDn      },
  { KEY_A1,        ksHome      },
  { KEY_HOME,      ksHome      },
  { KEY_C1,        ksEnd       },
  { KEY_END,       ksEnd       },
  { KEY_IC,        ksIns       },
  { KEY_DC,        ksDel       },
  { KEY_B2,        ksCenter    },
  { KEY_CANCEL,    ksCancel    },
  { KEY_COPY,      ksCopy      },
  { KEY_UNDO,      ksUndo      },
  { KEY_REDO,      ksRedo      },
  { KEY_OPEN,      ksOpen      },
  { KEY_CLOSE,     ksClose     },
  { KEY_COMMAND,   ksCommand   },
  { KEY_CREATE,    ksCreate    },
  { KEY_EXIT,      ksExit      },
  { KEY_FIND,      ksFind      },
  { KEY_HELP,      ksHelp      },
  { KEY_MARK,      ksMark      },
  { KEY_MESSAGE,   ksMessage   },
  { KEY_MOVE,      ksMove      },
  { KEY_NEXT,      ksNext      },
  { KEY_PREVIOUS,  ksPrevious  },
  { KEY_OPTIONS,   ksOptions   },
  { KEY_REFERENCE, ksReference },
  { KEY_REFRESH,   ksRefresh   },
  { KEY_REPLACE,   ksReplace   },
  { KEY_RESIZE,    ksResize    },
  { KEY_RESTART,   ksRestart   },
  { KEY_SUSPEND,   ksSuspend   },
  { KEY_RESUME,    ksResume    },
  { KEY_SAVE,      ksSave      },
  #ifdef XCURSES
  /* XCurses returns these for *shifted* keys (which is not wrong :-).
     However, we don't have key codes for shifted keys, but we can get
     the shift state via crt_getshiftstate(). Control-key combinations
     are obtained using crt_getshiftstate(). */
  { KEY_SLEFT,     ksLeft      },
  { KEY_SRIGHT,    ksRight     },
  { KEY_SHOME,     ksHome      },
  { KEY_SEND,      ksEnd       },
  { KEY_SDC,       ksDel       },
  { KEY_SIC,       ksIns       },
  #else
  { KEY_SLEFT,     ksCtrlLeft  },
  { KEY_SRIGHT,    ksCtrlRight },
  { KEY_SHOME,     ksCtrlHome  },
  { KEY_SEND,      ksCtrlEnd   },
  { KEY_SDC,       ksCtrlDel   },
  { KEY_SIC,       ksCtrlIns   },
  #endif
  #ifdef USE_PDCURSES
  { CTL_TAB,       ksCtrlTab   },
  { ALT_TAB,       ksAltTab    },
  { KEY_A2,        ksUp        },
  { KEY_B1,        ksLeft      },
  { KEY_B3,        ksRight     },
  { KEY_C2,        ksDown      },
  { CTL_LEFT,      ksCtrlLeft  },
  { CTL_RIGHT,     ksCtrlRight },
  { CTL_UP,        ksCtrlUp    },
  { CTL_DOWN,      ksCtrlDown  },
  { CTL_PGUP,      ksCtrlPgUp  },
  { CTL_PGDN,      ksCtrlPgDn  },
  { CTL_HOME,      ksCtrlHome  },
  { CTL_END,       ksCtrlEnd   },
  { CTL_INS,       ksCtrlIns   },
  { CTL_PADSTOP,   ksCtrlDel   },
  { CTL_PADCENTER, ksCtrlCentr },
  { ALT_LEFT,      ksAltLeft   },
  { ALT_RIGHT,     ksAltRight  },
  { ALT_UP,        ksAltUp     },
  { ALT_DOWN,      ksAltDown   },
  { ALT_PGUP,      ksAltPgUp   },
  { ALT_PGDN,      ksAltPgDn   },
  { ALT_HOME,      ksAltHome   },
  { ALT_END,       ksAltEnd    },
  { ALT_INS,       ksAltIns    },
  { ALT_DEL,       ksAltDel    },
  { ALT_ENTER,     ksAltEnter  },
  { ALT_PADENTER,  ksAltEnter  },
  { ALT_PADSTAR,   ksAltPStar  },
  { ALT_PADMINUS,  ksAltPMinus },
  { ALT_PADPLUS,   ksAltPPlus  },
  { ALT_ESC,       ksAltEsc    },
  { ALT_BKSP,      ksAltBkSp   },
  { ALT_MINUS,     ksAltMinus  },
  { ALT_EQUAL,     ksAltEqual  },
  { ALT_LBRACKET,  ksAltLBrack },
  { ALT_RBRACKET,  ksAltRBrack },
  { ALT_SEMICOLON, ksAltSemic  },
  { ALT_FQUOTE,    ksAltFQuote },
  { ALT_BQUOTE,    ksAltBQuote },
  { ALT_COMMA,     ksAltComma  },
  { ALT_STOP,      ksAltStop   },
  { ALT_FSLASH,    ksAltFSlash },
  { ALT_BSLASH,    ksAltBSlash },
  { ALT_0,         ksAlt0      },
  { ALT_1,         ksAlt1      },
  { ALT_2,         ksAlt2      },
  { ALT_3,         ksAlt3      },
  { ALT_4,         ksAlt4      },
  { ALT_5,         ksAlt5      },
  { ALT_6,         ksAlt6      },
  { ALT_7,         ksAlt7      },
  { ALT_8,         ksAlt8      },
  { ALT_9,         ksAlt9      },
  { ALT_A,         ksAltA      },
  { ALT_B,         ksAltB      },
  { ALT_C,         ksAltC      },
  { ALT_D,         ksAltD      },
  { ALT_E,         ksAltE      },
  { ALT_F,         ksAltF      },
  { ALT_G,         ksAltG      },
  { ALT_H,         ksAltH      },
  { ALT_I,         ksAltI      },
  { ALT_J,         ksAltJ      },
  { ALT_K,         ksAltK      },
  { ALT_L,         ksAltL      },
  { ALT_M,         ksAltM      },
  { ALT_N,         ksAltN      },
  { ALT_O,         ksAltO      },
  { ALT_P,         ksAltP      },
  { ALT_Q,         ksAltQ      },
  { ALT_R,         ksAltR      },
  { ALT_S,         ksAltS      },
  { ALT_T,         ksAltT      },
  { ALT_U,         ksAltU      },
  { ALT_V,         ksAltV      },
  { ALT_W,         ksAltW      },
  { ALT_X,         ksAltX      },
  { ALT_Y,         ksAltY      },
  { ALT_Z,         ksAltZ      },
  { KEY_F (11),    ksF11       },
  { KEY_F (12),    ksF12       },
  { KEY_F (23),    ksShF11     },
  { KEY_F (24),    ksShF12     },
  { KEY_F (35),    ksCtrlF11   },
  { KEY_F (36),    ksCtrlF12   },
  { KEY_F (47),    ksAltF11    },
  { KEY_F (48),    ksAltF12    },
  #endif
  { 0,               ksUnknown   }
};

#ifdef USE_PDCURSES
static TKeyTable AltGrFKeyTable [] =
{
  { ALT_0, ksAltGr0 },
  { ALT_1, ksAltGr1 },
  { ALT_2, ksAltGr2 },
  { ALT_3, ksAltGr3 },
  { ALT_4, ksAltGr4 },
  { ALT_5, ksAltGr5 },
  { ALT_6, ksAltGr6 },
  { ALT_7, ksAltGr7 },
  { ALT_8, ksAltGr8 },
  { ALT_9, ksAltGr9 },
  { ALT_A, ksAltGrA },
  { ALT_B, ksAltGrB },
  { ALT_C, ksAltGrC },
  { ALT_D, ksAltGrD },
  { ALT_E, ksAltGrE },
  { ALT_F, ksAltGrF },
  { ALT_G, ksAltGrG },
  { ALT_H, ksAltGrH },
  { ALT_I, ksAltGrI },
  { ALT_J, ksAltGrJ },
  { ALT_K, ksAltGrK },
  { ALT_L, ksAltGrL },
  { ALT_M, ksAltGrM },
  { ALT_N, ksAltGrN },
  { ALT_O, ksAltGrO },
  { ALT_P, ksAltGrP },
  { ALT_Q, ksAltGrQ },
  { ALT_R, ksAltGrR },
  { ALT_S, ksAltGrS },
  { ALT_T, ksAltGrT },
  { ALT_U, ksAltGrU },
  { ALT_V, ksAltGrV },
  { ALT_W, ksAltGrW },
  { ALT_X, ksAltGrX },
  { ALT_Y, ksAltGrY },
  { ALT_Z, ksAltGrZ },
  { 0,     0        }
};
#endif

static TKeyTable ShiftKeyTable [] =
{
  { ksIns, ksShIns },
  { ksDel, ksShDel },
  { 0,     0       }
};

static TKeyTable CtrlKeyTable [] =
{
  { ksLeft,   ksCtrlLeft  },
  { ksRight,  ksCtrlRight },
  { ksUp,     ksCtrlUp    },
  { ksDown,   ksCtrlDown  },
  { ksPgUp,   ksCtrlPgUp  },
  { ksPgDn,   ksCtrlPgDn  },
  { ksHome,   ksCtrlHome  },
  { ksEnd,    ksCtrlEnd   },
  { ksIns,    ksCtrlIns   },
  { ksDel,    ksCtrlDel   },
  { ksCenter, ksCtrlCentr },
  { ksF1,     ksCtrlF1    },
  { ksF2,     ksCtrlF2    },
  { ksF3,     ksCtrlF3    },
  { ksF4,     ksCtrlF4    },
  { ksF5,     ksCtrlF5    },
  { ksF6,     ksCtrlF6    },
  { ksF7,     ksCtrlF7    },
  { ksF8,     ksCtrlF8    },
  { ksF9,     ksCtrlF9    },
  { ksF10,    ksCtrlF10   },
  { ksF11,    ksCtrlF11   },
  { ksF12,    ksCtrlF12   },
  { 0,        0           }
};

static TKeyTable AltKeyTable [] =
{
  { ksLeft,   ksAltLeft  },
  { ksRight,  ksAltRight },
  { ksUp,     ksAltUp    },
  { ksDown,   ksAltDown  },
  { ksPgUp,   ksAltPgUp  },
  { ksPgDn,   ksAltPgDn  },
  { ksHome,   ksAltHome  },
  { ksEnd,    ksAltEnd   },
  { ksIns,    ksAltIns   },
  { ksDel,    ksAltDel   },
  { 0,        0          }
};

static TKeyTable AltGrKeyTable [] =
{
  { '0', ksAltGr0 },
  { '1', ksAltGr1 },
  { '2', ksAltGr2 },
  { '3', ksAltGr3 },
  { '4', ksAltGr4 },
  { '5', ksAltGr5 },
  { '6', ksAltGr6 },
  { '7', ksAltGr7 },
  { '8', ksAltGr8 },
  { '9', ksAltGr9 },
  { 'A', ksAltGrA },
  { 'B', ksAltGrB },
  { 'C', ksAltGrC },
  { 'D', ksAltGrD },
  { 'E', ksAltGrE },
  { 'F', ksAltGrF },
  { 'G', ksAltGrG },
  { 'H', ksAltGrH },
  { 'I', ksAltGrI },
  { 'J', ksAltGrJ },
  { 'K', ksAltGrK },
  { 'L', ksAltGrL },
  { 'M', ksAltGrM },
  { 'N', ksAltGrN },
  { 'O', ksAltGrO },
  { 'P', ksAltGrP },
  { 'Q', ksAltGrQ },
  { 'R', ksAltGrR },
  { 'S', ksAltGrS },
  { 'T', ksAltGrT },
  { 'U', ksAltGrU },
  { 'V', ksAltGrV },
  { 'W', ksAltGrW },
  { 'X', ksAltGrX },
  { 'Y', ksAltGrY },
  { 'Z', ksAltGrZ },
  { 0,   0        }
};

static TKeyTable ExtraKeyTable [] =
{
  { '0', ksExtra0 },
  { '1', ksExtra1 },
  { '2', ksExtra2 },
  { '3', ksExtra3 },
  { '4', ksExtra4 },
  { '5', ksExtra5 },
  { '6', ksExtra6 },
  { '7', ksExtra7 },
  { '8', ksExtra8 },
  { '9', ksExtra9 },
  { 'A', ksExtraA },
  { 'B', ksExtraB },
  { 'C', ksExtraC },
  { 'D', ksExtraD },
  { 'E', ksExtraE },
  { 'F', ksExtraF },
  { 'G', ksExtraG },
  { 'H', ksExtraH },
  { 'I', ksExtraI },
  { 'J', ksExtraJ },
  { 'K', ksExtraK },
  { 'L', ksExtraL },
  { 'M', ksExtraM },
  { 'N', ksExtraN },
  { 'O', ksExtraO },
  { 'P', ksExtraP },
  { 'Q', ksExtraQ },
  { 'R', ksExtraR },
  { 'S', ksExtraS },
  { 'T', ksExtraT },
  { 'U', ksExtraU },
  { 'V', ksExtraV },
  { 'W', ksExtraW },
  { 'X', ksExtraX },
  { 'Y', ksExtraY },
  { 'Z', ksExtraZ },
  { 0,   0       }
};

#ifndef USE_PDCURSES
static TKeyTable EscKeyTable [] =
{
  { KEY_BACKSPACE, ksAltBkSp   },
  { chBkSp,        ksAltBkSp   },
  { chTab,         ksAltTab    },
  { chLF,          ksAltEnter  },
  { chCR,          ksAltEnter  },
  { chEsc,         ksAltEsc    },
  { ' ',           ksAltSpace  },
  { '-',           ksAltMinus  },
  { '=',           ksAltEqual  },
  { '[',           ksAltLBrack },
  { ']',           ksAltRBrack },
  { ';',           ksAltSemic  },
  { '\'',          ksAltFQuote },
  { '`',           ksAltBQuote },
  { ',',           ksAltComma  },
  { '.',           ksAltStop   },
  { '/',           ksAltFSlash },
  { '\\',          ksAltBSlash },
  { 'A',           ksAltA      },
  { 'B',           ksAltB      },
  { 'C',           ksAltC      },
  { 'D',           ksAltD      },
  { 'E',           ksAltE      },
  { 'F',           ksAltF      },
  { 'G',           ksAltG      },
  { 'H',           ksAltH      },
  { 'I',           ksAltI      },
  { 'J',           ksAltJ      },
  { 'K',           ksAltK      },
  { 'L',           ksAltL      },
  { 'M',           ksAltM      },
  { 'N',           ksAltN      },
  { 'O',           ksAltO      },
  { 'P',           ksAltP      },
  { 'Q',           ksAltQ      },
  { 'R',           ksAltR      },
  { 'S',           ksAltS      },
  { 'T',           ksAltT      },
  { 'U',           ksAltU      },
  { 'V',           ksAltV      },
  { 'W',           ksAltW      },
  { 'X',           ksAltX      },
  { 'Y',           ksAltY      },
  { 'Z',           ksAltZ      },
  { '0',           ksAlt0      },
  { '1',           ksAlt1      },
  { '2',           ksAlt2      },
  { '3',           ksAlt3      },
  { '4',           ksAlt4      },
  { '5',           ksAlt5      },
  { '6',           ksAlt6      },
  { '7',           ksAlt7      },
  { '8',           ksAlt8      },
  { '9',           ksAlt9      },
  { 0,             0           }
};
#endif

#if !defined(USE_PDCURSES) || defined(XCURSES)
static chtype chars_0_31 [32] =
{
  ' ', 'O', 'O', '%',   0, '#', '*', 'o', 'o', 'o', 'o', '/',
  '+', 'd', 'A', '*',   0,   0,   0, '!', '#', '$'
};

static chtype chars_128_255 [128] =
{
  'C', 'u', 'e', 'a', 'a', 'a', 'a', 'c', 'e', 'e', 'e', 'i',
  'i', 'i', 'A', 'A', 'E', 'a', 'A', 'o', 'o', 'o', 'u', 'u',
  'y', 'O', 'U', 'C',   0, 'Y', 'P', 'f', 'a', 'i', 'o', 'u',
  'n', 'N',   0,   0, '?',   0,   0, '/', '/', '!',   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
  'a', 'B', 'T',   0, 'E', 'o', 'u', 't', 'O', 'O', 'O', 'O',
  'o', 'o', 'E', 'A', '=',   0,   0,   0, '/', '/', '%', '=',
    0, '.', '.', 'V', 'n', '2',   0, ' '
};
#endif

static void crt_setcbreak ()
{
  if (crt_CheckBreak)
    {
      noraw ();
      cbreak ();
    }
  else
    raw ();
}

static void crt_nodelay (bool bf)
{
  crt_setcbreak ();
  nodelay (crt_ActivePanel -> w, bf);
  nodelay (stdscr, bf);
}

static void crt_checkcheckbreak ()
{
  if (crt_CheckBreak != crt_LastCheckBreak)
    {
      crt_LastCheckBreak = crt_CheckBreak;
      crt_setcbreak ();
    }
}

void crt_setscroll (Boolean state);
void crt_setscroll (Boolean state)
{
  crt_ActivePanel -> ScrollState = state;
  scrollok (crt_ActivePanel -> w, state);
}

Boolean crt_getscroll (void);
Boolean crt_getscroll ()
{
  return crt_ActivePanel -> ScrollState;
}

TPanel crt_PanelAbove (TPanel Panel);
TPanel crt_PanelAbove (TPanel Panel)
{
  PANEL *p = panel_above (Panel ? Panel -> panel : NULL);
  return p ? (TPanel) panel_userptr (p) : NULL;
}

TPanel crt_PanelBelow (TPanel Panel);
TPanel crt_PanelBelow (TPanel Panel)
{
  PANEL *p = panel_below (Panel ? Panel -> panel : NULL);
  return p ? (TPanel) panel_userptr (p) : NULL;
}

static void crt_update ()
{
  int covered, x = 0, y = 0, setcursorflag = 0;
  TCursorShape shape;
  TPanel Panel, Panel2;
  update_panels ();
  for (Panel = crt_PanelBelow (NULL), shape = CursorIgnored;
       Panel && (shape = Panel -> CursorShape) == CursorIgnored;
       Panel = crt_PanelBelow (Panel));
  if (Panel)
    {
      getyx (Panel -> w, y, x);
      x += Panel -> LastWindMin & 0xff;
      y += Panel -> LastWindMin >> 8;
      for (covered = 0, Panel2 = crt_PanelAbove (Panel); !covered && Panel2; Panel2 = crt_PanelAbove (Panel2))
        if (x >= (Panel2 -> LastWindMin & 0xff) && x <= (Panel2 -> LastWindMax & 0xff) &&
            y >= (Panel2 -> LastWindMin >> 8)   && y <= (Panel2 -> LastWindMax >> 8)) covered = 1;
      if (covered)
        shape = CursorHidden;
      else
        setcursorflag = 1;
    }
  if (shape == CursorIgnored) shape = CursorHidden;
  if (shape != crt_last_shape)
    {
      crt_last_shape = shape;
      curs_set ((shape == CursorHidden) ? 0 :
                (shape == CursorNormal) ? 1 : 2);
    }
  if (shape == CursorHidden)
    SETCURSOR (0, 0);
  else if (setcursorflag)
    SETCURSOR (x, y);
  doupdate ();
}

/* Must be re-entrant! */
static void crt_refresh ()
{
  if (crt_refresh_inhibit == 0)
    {
      crt_refresh_inhibit++;
      crt_pending_refresh = FALSE;
      crt_update ();
      (void) idlok (crt_ActivePanel -> w, FALSE);
      crt_refresh_inhibit--;
    }
  else
    crt_refresh_flag++;
}

static void crt_need_refresh ()
{
  crt_checkcheckbreak ();
  if (crt_update_level >= UpdateAlways || crt_refresh_flag)
    {
      crt_refresh_flag = 0;
      crt_refresh ();
    }
  else if (crt_update_level >= UpdateRegularly && !crt_pending_refresh)
    {
      crt_pending_refresh = TRUE;
      crt_schedule_refresh ();
    }
}

void crt_PanelHide (TPanel Panel);
void crt_PanelHide (TPanel Panel)
{
  if (!Panel -> Hidden)
    {
      crt_refresh_inhibit++;
      hide_panel (Panel -> panel);
      Panel -> Hidden = TRUE;
      crt_refresh_inhibit--;
      crt_need_refresh ();
    }
}

void crt_PanelShow (TPanel Panel);
void crt_PanelShow (TPanel Panel)
{
  if (Panel -> Hidden)
    {
      crt_refresh_inhibit++;
      show_panel (Panel -> panel);
      Panel -> Hidden = FALSE;
      crt_refresh_inhibit--;
      crt_need_refresh ();
    }
}

Boolean crt_PanelHidden (TPanel Panel);
Boolean crt_PanelHidden (TPanel Panel)
{
  return Panel -> Hidden;
}

void crt_PanelTop (TPanel Panel);
void crt_PanelTop (TPanel Panel)
{
  if (Panel -> Hidden) crt_PanelShow (Panel);
  crt_refresh_inhibit++;
  top_panel (Panel -> panel);
  crt_refresh_inhibit--;
  crt_need_refresh ();
}

void crt_PanelBottom (TPanel Panel);
void crt_PanelBottom (TPanel Panel)
{
  if (Panel -> Hidden) crt_PanelShow (Panel);
  crt_refresh_inhibit++;
  bottom_panel (Panel -> panel);
  crt_refresh_inhibit--;
  crt_need_refresh ();
}

void crt_PanelMoveAbove (TPanel Panel, TPanel Above);
void crt_PanelMoveAbove (TPanel Panel, TPanel Above)
{
  if (!Above)
    crt_PanelBottom (Panel);
  else if (Above -> Hidden)
    crt_PanelHide (Panel);
  else if (Above != Panel)
    {
      PANEL *p, *p2;
      if (Panel -> Hidden) crt_PanelShow (Panel);
      crt_refresh_inhibit++;
      top_panel (Panel -> panel);
      p = panel_above (Above -> panel);
      while (p && p != Panel -> panel)
        {
          p2 = panel_above (p);
          top_panel (p);
          p = p2;
        }
      crt_refresh_inhibit--;
      crt_need_refresh ();
    }
}

void crt_PanelMoveBelow (TPanel Panel, TPanel Below);
void crt_PanelMoveBelow (TPanel Panel, TPanel Below)
{
  if (!Below)
    crt_PanelTop (Panel);
  else if (Below -> Hidden)
    crt_PanelHide (Panel);
  else if (Below != Panel)
    {
      PANEL *p, *p2;
      if (Panel -> Hidden) crt_PanelShow (Panel);
      crt_refresh_inhibit++;
      bottom_panel (Panel -> panel);
      p = panel_below (Below -> panel);
      while (p && p != Panel -> panel)
        {
          p2 = panel_below (p);
          bottom_panel (p);
          p = p2;
        }
      crt_refresh_inhibit--;
      crt_need_refresh ();
    }
}

void crt_PanelBindToBackground (TPanel Panel, Boolean Bind);
void crt_PanelBindToBackground (TPanel Panel, Boolean Bind)
{
  Panel -> BoundToBackground = Bind;
}

Boolean crt_PanelIsBoundToBackground (TPanel Panel);
Boolean crt_PanelIsBoundToBackground (TPanel Panel)
{
  return Panel -> BoundToBackground;
}

void crt_setcursorshape (TCursorShape shape);
void crt_setcursorshape (TCursorShape shape)
{
  crt_ActivePanel -> CursorShape = shape;
  crt_need_refresh ();
}

TCursorShape crt_getcursorshape (void);
TCursorShape crt_getcursorshape ()
{
  return crt_ActivePanel -> CursorShape;
}

TPanel crt_GetActivePanel (void);
TPanel crt_GetActivePanel ()
{
  return crt_ActivePanel;
}

static WINDOW *crt_newwin (int x1, int y1, int x2, int y2, Boolean BoundToBackground)
{
  int xsize = x2 - x1 + 1, ysize = y2 - y1 + 1;
  WINDOW *w = newwin (ysize, xsize, y1, x1);
  if (!w) crt_fatal (1);
  if (BoundToBackground)
    {
      chtype Buffer [xsize + 1];
      int yc;
      for (yc = 0; yc < ysize; yc++)
        {
          mvwinchnstr (stdscr, yc + y1, x1, Buffer, xsize);
          mvwaddchnstr (w, yc, 0, Buffer, xsize);
        }
    }
  keypad (w, TRUE);
  leaveok (w, FALSE);
  crt_last_shape = - 1;
  return w;
}

/* Save contents of the panel to stdscr to imitate BP behaviour */
static void savetostdscr (TPanel Panel)
{
  if (Panel -> BoundToBackground)
    {
      TPanel p;
      int yc,
          LastXMin = Panel -> LastWindMin & 0xff,
          LastYMin = Panel -> LastWindMin >> 8,
          LastXSize = (Panel -> LastWindMax & 0xff) - LastXMin + 1,
          LastYSize = (Panel -> LastWindMax >> 8) - LastYMin + 1;
      chtype Buffer [LastXSize + 1];
      for (yc = 0; yc < LastYSize; yc++)
        {
          mvwinchnstr (Panel -> w, yc, 0, Buffer, LastXSize);
          mvwaddchnstr (stdscr, yc + LastYMin, LastXMin, Buffer, LastXSize);
        }
      wnoutrefresh (stdscr);
      for (p = crt_PanelList; p; p = p -> Next) touchwin (p -> w);
    }
}

static void win_changed ()
{
  int x1, y1, x2, y2, x, y;
  WINDOW *oldwin = crt_ActivePanel -> w;
  crt_refresh_inhibit++;
  x2 = WindXMax; if (x2 < 0) x2 = 0; if (x2 >= crt_ScreenSize.x) x2 = crt_ScreenSize.x - 1;
  y2 = WindYMax; if (y2 < 0) y2 = 0; if (y2 >= crt_ScreenSize.y) y2 = crt_ScreenSize.y - 1;
  x1 = WindXMin; if (x1 < 0) x1 = 0; if (x1 > x2) x1 = x2;
  y1 = WindYMin; if (y1 < 0) y1 = 0; if (y1 > y2) y1 = y2;
  getyx (crt_ActivePanel -> w, y, x);
  x += crt_ActivePanel -> LastWindMin & 0xff;
  y += crt_ActivePanel -> LastWindMin >> 8;
  savetostdscr (crt_ActivePanel);
  crt_ActivePanel -> w = crt_newwin (x1, y1, x2, y2, crt_ActivePanel -> BoundToBackground);
  replace_panel (crt_ActivePanel -> panel, crt_ActivePanel -> w);
  delwin (oldwin);
  crt_last_shape = - 1;
  crt_setscroll (crt_ActivePanel -> ScrollState);
  if (x < x1) x = x1;
  if (x > x2) x = x2;
  if (y < y1) y = y1;
  if (y > y2) y = y2;
  wmove (crt_ActivePanel -> w, y - y1, x - x1);
  crt_ActivePanel -> LastWindMin = crt_WindMin = x1 + (y1 << 8);
  crt_ActivePanel -> LastWindMax = crt_WindMax = x2 + (y2 << 8);
  crt_refresh_inhibit--;
  crt_need_refresh ();
}

static void GetLastMode ()
{
  if (crt_HasColors)
    {
      crt_LastMode = 0;
      if (crt_ColorFlag)          crt_LastMode += 1;
      if (crt_ScreenSize.x >= 80) crt_LastMode += 2;
    }
  else
    crt_LastMode = 7;
  if (crt_ScreenSize.y >= 43) crt_LastMode += 0x100;
}

static void check_winchanged ()
{
  if (crt_screen_size_changed)
    {
      crt_refresh_inhibit++;
      napms (100); /* wait for the new unimap to be loaded */
      napms (100); /* try again in case the first napms() was interrupted by a signal */
      #ifndef USE_PDCURSES
      endwin ();
      #endif
      refresh ();
      getmaxyx (stdscr, crt_ScreenSize.y, crt_ScreenSize.x);
      crt_screen_size_changed = 0;
      crt_refresh_inhibit--;
      win_changed ();
      GetLastMode ();
      if (crt_key_buf == 0) crt_key_buf = KEY_RESIZE;
    }
  else if (crt_WindMin != crt_ActivePanel -> LastWindMin || crt_WindMax != crt_ActivePanel -> LastWindMax)
    win_changed ();
}

void crt_PanelActivate (TPanel Panel);
void crt_PanelActivate (TPanel Panel)
{
  if (crt_ActivePanel)
    {
      check_winchanged ();
      crt_ActivePanel -> TextAttr = crt_TextAttr;
      crt_ActivePanel -> WindMin  = crt_WindMin;
      crt_ActivePanel -> WindMax  = crt_WindMax;
    }
  crt_ActivePanel = Panel;
  crt_TextAttr = crt_ActivePanel -> TextAttr;
  crt_WindMin = crt_ActivePanel -> WindMin;
  crt_WindMax = crt_ActivePanel -> WindMax;
}

void crt_PanelNew (int x1, int y1, int x2, int y2, Boolean BindToBackground);
void crt_PanelNew (int x1, int y1, int x2, int y2, Boolean BindToBackground)
{
  TPanel NewPanel = malloc (sizeof (TPanelData));
  crt_refresh_inhibit++;
  NewPanel -> Next = crt_PanelList;
  crt_PanelList = NewPanel;
  if (x2 < 1) x2 = 1; if (x2 > crt_ScreenSize.x) x2 = crt_ScreenSize.x;
  if (y2 < 1) y2 = 1; if (y2 > crt_ScreenSize.y) y2 = crt_ScreenSize.y;
  if (x1 < 1) x1 = 1; if (x1 > x2) x1 = x2;
  if (y1 < 1) y1 = 1; if (y1 > y2) y1 = y2;
  NewPanel -> BoundToBackground = BindToBackground;
  NewPanel -> Hidden = FALSE;
  NewPanel -> ScrollState     = crt_ActivePanel ? crt_ActivePanel -> ScrollState     : TRUE;
  NewPanel -> PCCharSet       = crt_ActivePanel ? crt_ActivePanel -> PCCharSet       : TRUE;
  NewPanel -> UseControlChars = crt_ActivePanel ? crt_ActivePanel -> UseControlChars : TRUE;
  NewPanel -> CursorShape     = crt_ActivePanel ? crt_ActivePanel -> CursorShape     : CursorNormal;
  NewPanel -> TextAttr        = crt_TextAttr;
  NewPanel -> w = crt_newwin (x1 - 1, y1 - 1, x2 - 1, y2 - 1, NewPanel -> BoundToBackground);
  NewPanel -> panel = new_panel (NewPanel -> w);
  set_panel_userptr (NewPanel -> panel, (void *) NewPanel);
  NewPanel -> LastWindMin = NewPanel -> WindMin = (x1 - 1) + ((y1 - 1) << 8);
  NewPanel -> LastWindMax = NewPanel -> WindMax = (x2 - 1) + ((y2 - 1) << 8);
  crt_PanelActivate (NewPanel);
  wmove (NewPanel -> w, 0, 0);
  crt_setscroll (NewPanel -> ScrollState);
  crt_refresh_inhibit--;
  crt_need_refresh ();
}

void crt_PanelDelete (TPanel Panel);
void crt_PanelDelete (TPanel Panel)
{
  TPanel *PList = &crt_PanelList;
  while (*PList && *PList != Panel) PList = &((*PList) -> Next);
  if (!*PList) crt_fatal (2);
  crt_refresh_inhibit++;
  *PList = Panel -> Next;
  if (!crt_PanelList) crt_fatal (3);
  savetostdscr (Panel);
  del_panel (Panel -> panel);
  delwin (Panel -> w);
  free (Panel);
  if (Panel == crt_ActivePanel)
    {
      crt_ActivePanel = NULL;
      crt_PanelActivate (crt_PanelList);
    }
  crt_last_shape = - 1;
  crt_refresh_inhibit--;
  crt_need_refresh ();
}

static void rawout (Boolean flag)
{
  #ifdef HAVE_RAW_OUTPUT
  raw_output (flag);
  #else
  (void) flag;
  #endif
}

static int attr2cursattr (TTextAttr attr)
{
  #ifdef XCURSES
  /* PDCurses under X has serious problems writing black on black. :-(
     So we write some other color on black and output spaces (see chtransform()) */
  if ((attr & ~0x88) == 0) attr = 7;
  #endif
  return (crt_ColorFlag ? crt_attrs : crt_monoattrs) [attr & 7] [(attr >> 4) & 7] |
         (((attr & 8) && (crt_ColorFlag || (attr & 0x77))) ? A_BOLD : 0) |
         ((attr & 0x80) ? A_BLINK : 0);
}

static TTextAttr cursattr2attr (int cursattr)
{
  int attr = 0;
  cursattr &= (A_ATTRIBUTES & ~A_ALTCHARSET);
  if (attr2cursattr (crt_NormAttr) == cursattr) return crt_NormAttr;
  while (attr < 0x100 && attr2cursattr (attr) != cursattr) attr++;
  return (attr == 0x100) ? crt_NormAttr : attr;
}

void crt_setattr (void);
void crt_setattr ()
{
  int cursattr;
  check_winchanged ();
  cursattr = attr2cursattr (crt_TextAttr);
  wattrset (crt_ActivePanel -> w, cursattr);
  wbkgdset (crt_ActivePanel -> w, cursattr | ' ');
}

static void crt_set_typeahead ()
{
  typeahead ((crt_update_level >= UpdateInput) ? - 1 : crt_get_input_fd ());
}

void crt_update_immediately (void);
void crt_update_immediately ()
{
  crt_setattr ();
  if (crt_update_level == UpdateRegularly) crt_stop_refresh ();
  typeahead (- 1);
  crt_refresh ();
  crt_set_typeahead ();
}

void crt_redraw (void);
void crt_redraw ()
{
  TPanel p;
  for (p = crt_PanelList; p; p = p -> Next) touchwin (p -> w);
  clearok (curscr, TRUE);
  crt_last_shape = - 1;
  crt_update ();
}

void crt_setupdatelevel (TCRTUpdate level);
void crt_setupdatelevel (TCRTUpdate level)
{
  if (crt_update_level == UpdateRegularly) crt_update_immediately ();
  crt_update_level = level;
  crt_set_typeahead ();
}

void crt_restore_terminal_no_crt (void);
void crt_restore_terminal_no_crt ()
{
  if (!crt_inited || crt_terminal_no_crt) return;
  if (crt_ActivePanel) crt_update_immediately ();
  crt_terminal_no_crt = TRUE;
  endwin ();
  /* reset_shell_mode (); */
  #ifdef SIGTSTP
  if (!crt_old_tstp_handler) crt_old_tstp_handler = signal (SIGTSTP, SIG_DFL);
  #endif
}

void crt_restore_terminal_crt (void);
void crt_restore_terminal_crt ()
{
  if (!crt_inited || !crt_terminal_no_crt) return;
  #ifdef SIGTSTP
  if (crt_old_tstp_handler)
    {
      signal (SIGTSTP, crt_old_tstp_handler);
      crt_old_tstp_handler = NULL;
    }
  #endif
  /* reset_prog_mode (); */
  rawout (TRUE);
  refresh ();
  #ifdef XCURSES
  PDC_save_key_modifiers (TRUE);
  #endif
  crt_terminal_no_crt = FALSE;
  clearok (curscr, TRUE);
  crt_last_shape = - 1;
  crt_need_refresh ();
}

void crt_clearterminal (void);
void crt_clearterminal ()
{
  crt_restore_terminal_no_crt ();
  clrscr ();
}

void crt_done (void);
void crt_done ()
{
  if (!crt_inited) return;
  crt_special_done ();
  if (crt_ActivePanel && !crt_terminal_no_crt)
    {
      crt_update_immediately ();
      endwin ();
    }
  crt_inited = 0;
  #ifdef XCURSES
  XCursesExit ();
  #endif
}

void crt_set_curses_mode (Boolean on);
void crt_set_curses_mode (Boolean on)
{
  if (on)
    {
      reset_prog_mode ();
      rawout (TRUE);
    }
  else
    {
      rawout (FALSE);
      reset_shell_mode ();
    }
}

void crt_init (void);
void crt_init ()
{
  int fg, bg, c, cursattr;
  if (crt_inited) return;
  #ifdef XCURSES
  XCursesProgramName = crt_get_program_name ();
  #endif
  if (!initscr ())
    {
      fprintf (stderr, "could not init curses");
      exit (1);
    }
  #ifdef XCURSES
  PDC_save_key_modifiers (TRUE);
  #endif
  crt_inited = 1;
  #ifndef USE_PDCURSES
  crt_linux_console = strncmp (termname (), "linux", 5) == 0;
  #endif
  #if !defined(USE_PDCURSES) || defined(XCURSES)
  chars_0_31 [ 4] = ACS_DIAMOND;
  chars_0_31 [16] = ACS_RARROW;
  chars_0_31 [17] = ACS_LARROW;
  chars_0_31 [18] = ACS_VLINE;
  chars_0_31 [22] = ACS_S9;
  chars_0_31 [23] = ACS_DARROW;
  chars_0_31 [24] = ACS_UARROW;
  chars_0_31 [25] = ACS_DARROW;
  chars_0_31 [26] = ACS_RARROW;
  chars_0_31 [27] = ACS_LARROW;
  chars_0_31 [28] = ACS_LLCORNER;
  chars_0_31 [29] = ACS_HLINE;
  chars_0_31 [30] = ACS_UARROW;
  chars_0_31 [31] = ACS_DARROW;
  chars_128_255 [ 28] = ACS_STERLING;
  chars_128_255 [ 38] = ACS_DEGREE;
  chars_128_255 [ 39] = ACS_DEGREE;
  chars_128_255 [ 41] = ACS_ULCORNER;
  chars_128_255 [ 42] = ACS_URCORNER;
  chars_128_255 [ 46] = ACS_LARROW;
  chars_128_255 [ 47] = ACS_RARROW;
  chars_128_255 [ 48] = ACS_CKBOARD;
  chars_128_255 [ 49] = ACS_CKBOARD;
  chars_128_255 [ 50] = ACS_CKBOARD;
  chars_128_255 [ 51] = ACS_VLINE;
  chars_128_255 [ 52] = ACS_RTEE;
  chars_128_255 [ 53] = ACS_RTEE;
  chars_128_255 [ 54] = ACS_RTEE;
  chars_128_255 [ 55] = ACS_URCORNER;
  chars_128_255 [ 56] = ACS_URCORNER;
  chars_128_255 [ 57] = ACS_RTEE;
  chars_128_255 [ 58] = ACS_VLINE;
  chars_128_255 [ 59] = ACS_URCORNER;
  chars_128_255 [ 60] = ACS_LRCORNER;
  chars_128_255 [ 61] = ACS_LRCORNER;
  chars_128_255 [ 62] = ACS_LRCORNER;
  chars_128_255 [ 63] = ACS_URCORNER;
  chars_128_255 [ 64] = ACS_LLCORNER;
  chars_128_255 [ 65] = ACS_BTEE;
  chars_128_255 [ 66] = ACS_TTEE;
  chars_128_255 [ 67] = ACS_LTEE;
  chars_128_255 [ 68] = ACS_HLINE;
  chars_128_255 [ 69] = ACS_PLUS;
  chars_128_255 [ 70] = ACS_LTEE;
  chars_128_255 [ 71] = ACS_LTEE;
  chars_128_255 [ 72] = ACS_LLCORNER;
  chars_128_255 [ 73] = ACS_ULCORNER;
  chars_128_255 [ 74] = ACS_BTEE;
  chars_128_255 [ 75] = ACS_TTEE;
  chars_128_255 [ 76] = ACS_LTEE;
  chars_128_255 [ 77] = ACS_HLINE;
  chars_128_255 [ 78] = ACS_PLUS;
  chars_128_255 [ 79] = ACS_BTEE;
  chars_128_255 [ 80] = ACS_BTEE;
  chars_128_255 [ 81] = ACS_TTEE;
  chars_128_255 [ 82] = ACS_TTEE;
  chars_128_255 [ 83] = ACS_LLCORNER;
  chars_128_255 [ 84] = ACS_LLCORNER;
  chars_128_255 [ 85] = ACS_ULCORNER;
  chars_128_255 [ 86] = ACS_ULCORNER;
  chars_128_255 [ 87] = ACS_PLUS;
  chars_128_255 [ 88] = ACS_PLUS;
  chars_128_255 [ 89] = ACS_LRCORNER;
  chars_128_255 [ 90] = ACS_ULCORNER;
  chars_128_255 [ 91] = ACS_BLOCK;
  chars_128_255 [ 92] = ACS_BLOCK;
  chars_128_255 [ 93] = ACS_BLOCK;
  chars_128_255 [ 94] = ACS_BLOCK;
  chars_128_255 [ 95] = ACS_BLOCK;
  chars_128_255 [ 99] = ACS_PI;
  chars_128_255 [113] = ACS_PLMINUS;
  chars_128_255 [114] = ACS_GEQUAL;
  chars_128_255 [115] = ACS_LEQUAL;
  chars_128_255 [120] = ACS_DEGREE;
  chars_128_255 [126] = ACS_BULLET;
  #endif
  cbreak ();
  noecho ();
  scrollok (stdscr, TRUE);
  rawout (TRUE);
  crt_ColorFlag = crt_HasColors = has_colors ();
  if (crt_ColorFlag)
    {
      start_color ();
      c = 0;
      for (bg = 0; bg < 8; bg++)
        for (fg = 0; fg < 8; fg++)
          if (INVIS_WORKS && bg == fg && bg > 0)
            crt_attrs [fg] [bg] = crt_attrs [0] [bg] | A_INVIS;
          else if (REVERSE_WORKS && fg < bg)
            crt_attrs [fg] [bg] = crt_attrs [bg] [fg] | A_REVERSE;
          else
            {
              if (init_pair (++c, crt_colors [fg], crt_colors [bg]) == ERR)
                {
                  crt_done ();
                  fprintf (stderr, "could not create color pair (%i,%i)", fg, bg);
                  exit (1);
                }
              crt_attrs [fg] [bg] = COLOR_PAIR (c);
            }
      for (bg = 0; bg < 8; bg++)
        for (fg = 0; fg < 8; fg++)
          crt_monoattrs [fg] [bg] = crt_attrs [7] [0];
      crt_monoattrs [0] [0] = crt_attrs [0] [0];
      crt_monoattrs [1] [0] = crt_attrs [1] [0];
      crt_monoattrs [0] [7] = crt_attrs [0] [7];
    }
  else
    {
      for (bg = 0; bg < 8; bg++)
        for (fg = 0; fg < 8; fg++)
          crt_monoattrs [fg] [bg] = A_NORMAL;
      crt_monoattrs [0] [0] = A_INVIS;
      crt_monoattrs [1] [0] = A_UNDERLINE;
      crt_monoattrs [0] [7] = A_REVERSE;
    }
  cursattr = attr2cursattr (crt_NormAttr);
  attrset (cursattr);
  bkgdset (cursattr);
  erase ();
  crt_special_init ();
  keypad (stdscr, TRUE);
  getmaxyx (stdscr, crt_ScreenSize.y, crt_ScreenSize.x);
  GetLastMode ();
  crt_TextAttr = crt_NormAttr;
  crt_PanelNew (1, 1, crt_ScreenSize.x, crt_ScreenSize.y, TRUE);
  typeahead (- 1);
  crt_refresh (); /* prevents flickering at the beginning of XCurses programs */
  crt_setupdatelevel (UpdateRegularly);
}

void crt_changetextmode (Boolean Columns40, Boolean Lines50);
void crt_changetextmode (Boolean Columns40, Boolean Lines50)
{
  if (crt_update_level == UpdateRegularly) crt_stop_refresh ();
  crt_refresh ();
  crt_settextmode (Columns40, Lines50);
  napms (1);
  crt_setattr ();
  crt_redraw ();
  GetLastMode ();
  if (crt_key_buf == KEY_RESIZE) crt_key_buf = 0;
}

void crt_delay (unsigned ms);
void crt_delay (unsigned ms)
{
  crt_update_immediately ();
  napms (ms);
}

static int wgetk ()
{
  int i;
  if (crt_screen_size_changed)
    i = ERR;
  else
    i = wgetch (stdscr); /* use stdscr rather than crt_ActivePanel -> w, so a non-top panel will not be redrawn */
  if (i == KEY_RESIZE)
    crt_screen_size_changed++;
  if (crt_screen_size_changed)
    {
      check_winchanged ();
      if (i != KEY_RESIZE && i != ERR) crt_key_buf = i;
      i = KEY_RESIZE;
    }
  return i;
}

void crt_select (void *PrivateData, Boolean *ReadSelect, Boolean *WriteSelect, Boolean *ExceptSelect);
void crt_select (void *PrivateData, Boolean *ReadSelect, Boolean *WriteSelect, Boolean *ExceptSelect)
{
  (void) PrivateData;
  if (*ReadSelect)
    {
      if (crt_update_level == UpdateRegularly)
        {
          crt_stop_refresh ();
          crt_refresh ();
        }
      (*ReadSelect) = !!(crt_unget_key_buf || crt_key_buf || crt_fkey_buf);
    }
  if (crt_get_output_fd () >= 0) (*WriteSelect) = FALSE;
  (*ExceptSelect) = FALSE;
}

void crt_ungetch (TKey ch);
void crt_ungetch (TKey ch)
{
  crt_unget_key_buf = ch;
}

Boolean crt_keypressed (void);
Boolean crt_keypressed ()
{
  int ch;
  if (crt_unget_key_buf || crt_key_buf || crt_fkey_buf) return 1;
  crt_nodelay (TRUE);
  ch = wgetk ();
  if (ch == ERR) return 0;
  crt_key_buf = ch;
  return 1;
}

static Char find_key (TKeyTable *table, int ch)
{
  while (table -> curses_key != 0 && table -> curses_key != ch) table++;
  return table -> crt_key;
}

/* Don't use toupper() since it messes up the function key codes */
static int upcase (int ch)
{
  if (ch >= 'a' && ch <= 'z')
    return ch - 'a' + 'A';
  else
    return ch;
}

TKey crt_readkeyword (void);
TKey crt_readkeyword ()
{
  int ch, shiftstate;
  Char c, cs;
  if (crt_update_level >= UpdateWaitInput) crt_update_immediately ();
  if (crt_unget_key_buf)
    {
      TKey k = crt_unget_key_buf;
      crt_unget_key_buf = 0;
      return k;
    }
  else if (crt_key_buf)
    {
      ch = crt_key_buf;
      crt_key_buf = 0;
    }
  else
    {
      crt_nodelay (FALSE);
      do
        {
          ch = wgetk ();
          if (crt_unget_key_buf)
            {
              TKey k = crt_unget_key_buf;
              crt_unget_key_buf = 0;
              if (ch != ERR) crt_key_buf = ch;
              return k;
            }
        }
      while (ch == ERR);
    }
  #ifndef USE_PDCURSES
  if (ch == chEsc)
    {
      crt_nodelay (TRUE);
      ch = wgetk ();
      if (ch == ERR) return chEsc;
      c = find_key (EscKeyTable, upcase (ch));
      if (c != 0)
        return 0x100 * c;
      if (ch >= KEY_F (1) && ch <= KEY_F (10))
        return 0x100 * (ksAltF1 + (ch - KEY_F (1)));
      if (ch >= KEY_F (11) && ch <= KEY_F (12))
        return 0x100 * (ksAltF11 + (ch - KEY_F (11)));
      crt_key_buf = ch;
      return chEsc;
    }
  #endif
  shiftstate = crt_getshiftstate ();
  c = find_key (KeyTable, ch);
  if (c != 0) return c;
  if (ch == chTab && (shiftstate & shShift)) return 0x100 * ksShTab;
  #ifdef USE_PDCURSES
  if ((shiftstate & shAltGr) && (cs = find_key (AltGrFKeyTable, ch)))
    return 0x100 * cs;
  #endif
  if ((shiftstate & shAltGr) && (cs = find_key (AltGrKeyTable, upcase (ch)))) return 0x100 * cs;
  if ((shiftstate & shExtra) && (cs = find_key (ExtraKeyTable, upcase (ch)))) return 0x100 * cs;
  if (1 <= ch && ch <= 0xff) return ch;
  if (ch >= KEY_F (1) && ch <= KEY_F (10))
    {
      if (shiftstate & shAnyAlt)
        c = ksAltF1 + (ch - KEY_F (1));
      else if (shiftstate & shCtrl)
        c = ksCtrlF1 + (ch - KEY_F (1));
      else if (shiftstate & shShift)
        c = ksShF1 + (ch - KEY_F (1));
      else
        c = ksF1 + (ch - KEY_F (1));
    }
  else if (ch >= KEY_F (FKEYSH) && ch <= KEY_F (FKEYSH + 9))
    c = ksShF1 + (ch - KEY_F (FKEYSH));
  else if (ch >= KEY_F (FKEYCTRL) && ch <= KEY_F (FKEYCTRL + 9))
    c = ksCtrlF1 + (ch - KEY_F (FKEYCTRL));
  else if (ch >= KEY_F (FKEYALT) && ch <= KEY_F (FKEYALT + 9))
    c = ksAltF1 + (ch - KEY_F (FKEYALT));
  else
    c = find_key (FKeyTable, ch);
  if ((shiftstate & shShift) && (cs = find_key (ShiftKeyTable, c)))
    c = cs;
  else if ((shiftstate & shCtrl) && (cs = find_key (CtrlKeyTable, c)))
    c = cs;
  else if ((shiftstate & shAlt) && (cs = find_key (AltKeyTable, c)))
    c = cs;
  return 0x100 * c;
}

Char crt_readkey (void);
Char crt_readkey ()
{
  Char tmp;
  int ch;
  if (crt_fkey_buf)
    {
      tmp = crt_fkey_buf;
      crt_fkey_buf = 0;
      return tmp;
    }
  ch = crt_readkeyword ();
  if (ch & 0xff) return ch;
  crt_fkey_buf= ch / 0x100;
  return 0;
}

void crt_gotoxy (int x, int y);
void crt_gotoxy (int x, int y)
{
  crt_setattr ();
  wmove (crt_ActivePanel -> w, y - 1, x - 1);
  crt_need_refresh ();
}

int crt_wherex (void);
int crt_wherex ()
{
  int x, y;
  crt_setattr ();
  getyx (crt_ActivePanel -> w, y, x);
  return x + 1;
}

int crt_wherey (void);
int crt_wherey ()
{
  int x, y;
  crt_setattr ();
  getyx (crt_ActivePanel -> w, y, x);
  return y + 1;
}

size_t crt_winsize (void);
size_t crt_winsize ()
{
  return (WindXMax - WindXMin + 1) * (WindYMax - WindYMin + 1) * sizeof (chtype);
}

void crt_readwin (chtype *buf);
void crt_readwin (chtype *buf)
{
  int xsize = WindXMax - WindXMin + 1, ysize = WindYMax - WindYMin + 1, yc, sx, sy;
  chtype temp [xsize + 1];
  crt_setattr ();
  getyx (crt_ActivePanel -> w, sy, sx);
  for (yc = 0; yc < ysize; yc++)
    {
      mvwinchnstr (crt_ActivePanel -> w, yc, 0, temp, xsize);
      memcpy (buf + xsize * yc, temp, xsize * sizeof (chtype)); /* don't copy the 0 terminator! */
    }
  wmove (crt_ActivePanel -> w, sy, sx);
}

void crt_writewin (chtype *buf);
void crt_writewin (chtype *buf)
{
  int xsize = WindXMax - WindXMin + 1, ysize = WindYMax - WindYMin + 1, yc, sx, sy;
  crt_setattr ();
  getyx (crt_ActivePanel -> w, sy, sx);
  for (yc = 0; yc < ysize; yc++)
    mvwaddchnstr (crt_ActivePanel -> w, yc, 0, buf + xsize * yc, xsize);
  wmove (crt_ActivePanel -> w, sy, sx);
  crt_need_refresh ();
}

void crt_clrscr (void);
void crt_clrscr ()
{
  crt_setattr ();
  werase (crt_ActivePanel -> w);
  crt_need_refresh ();
}

void crt_clreol (void);
void crt_clreol ()
{
  crt_setattr ();
  wclrtoeol (crt_ActivePanel -> w);
  crt_need_refresh ();
}

void crt_insline (void);
void crt_insline ()
{
  crt_setattr ();
  winsertln (crt_ActivePanel -> w);
  (void) idlok (crt_ActivePanel -> w, TRUE);
  crt_need_refresh ();
}

void crt_delline (void);
void crt_delline ()
{
  crt_setattr ();
  wdeleteln (crt_ActivePanel -> w);
  (void) idlok (crt_ActivePanel -> w, TRUE);
  crt_need_refresh ();
}

void crt_setpccharset (Boolean PCCharSet);
void crt_setpccharset (Boolean PCCharSet)
{
  crt_ActivePanel -> PCCharSet = PCCharSet;
}

Boolean crt_getpccharset (void);
Boolean crt_getpccharset ()
{
  return crt_ActivePanel -> PCCharSet;
}

void crt_setcontrolchars (Boolean UseControlChars);
void crt_setcontrolchars (Boolean UseControlChars)
{
  crt_ActivePanel -> UseControlChars = UseControlChars;
}

Boolean crt_getcontrolchars (void);
Boolean crt_getcontrolchars ()
{
  return crt_ActivePanel -> UseControlChars;
}

size_t crt_read (void *PrivateData, Char *buffer, size_t size);
size_t crt_read (void *PrivateData, Char *buffer, size_t size)
{
  int res;
  size_t n;
  Char *p;
  (void) PrivateData;
  if (!crt_line_buf_count)
    {
      crt_nodelay (FALSE);
      rawout (FALSE);
      echo ();
      if (crt_update_level >= UpdateWaitInput) crt_update_immediately ();
      do
        {
          errno = 0;
          res = wgetnstr (crt_ActivePanel -> w, crt_line_buf, MAXLENGTH - 1);
        }
      while
        #ifdef EINTR
        (res == ERR && errno == EINTR);
        #else
        (0);
        #endif
      if (res == ERR)
        crt_line_buf_count = 0;
      else
        {
          if (crt_CheckEOF)
            {
              do
                {
                  p = strchr (crt_line_buf, 26);
                  if (p) *p = 0;
                }
              while (p);
            }
          crt_line_buf_count = strlen (crt_line_buf);
          crt_line_buf [crt_line_buf_count++] = '\n';
        }
      crt_line_buf_pos = crt_line_buf;
      noecho ();
      rawout (TRUE);
    }
  n = (crt_line_buf_count < size) ? crt_line_buf_count : size;
  memcpy (buffer, crt_line_buf_pos, n);
  crt_line_buf_pos += n;
  crt_line_buf_count -= n;
  return n;
}

void crt_flash (void);
void crt_flash ()
{
  flash ();
}

void crt_beep (void);
void crt_beep ()
{
  if (crt_VisualBell)
    crt_flash ();
  else
    {
      beep ();
      crt_delay (100);
    }
}

static chtype chtransform (Char ch, TTextAttr attr)
{
  Boolean pccs = crt_ActivePanel -> PCCharSet || (ch < ' ');
  if (ch == 0
  #ifdef XCURSES
    || (crt_ColorFlag && (attr & 7) == ((attr >> 4) & 7))
  #endif
    || (attr & ~(crt_ColorFlag ? 0x80 : 0x88)) == 0
    ) return ' ';
  #if !defined(USE_PDCURSES) || defined(XCURSES)
  if ((!crt_linux_console && pccs)
      || ch == chBell || ch == chBkSp || ch == chTab || ch == chLF || ch == chFF
      || ch == chCR || ch == chEsc || ch == 14 || ch == 15 || ch == 155)
    {
      if (ch < 32) return chars_0_31 [ch];
      if (ch >= 128) return chars_128_255 [ch - 128];
    }
  #endif
  if (!pccs && iscntrl (ch)) return ' ';
  return (crt_linux_console && pccs) ? ch | A_ALTCHARSET : ch;
}

void crt_readchar (int x, int y, Char *ch, TTextAttr *attr);
void crt_readchar (int x, int y, Char *ch, TTextAttr *attr)
{
  int sx, sy, cc;
  chtype c;
  crt_setattr ();
  getyx (crt_ActivePanel -> w, sy, sx);
  c = mvwinch (crt_ActivePanel -> w, y - 1, x - 1);
  *attr = cursattr2attr (c);
  c &= A_CHARTEXT;
  for (cc = 0x20; cc < 0x120 && (chtransform (cc % 0x100, 7) & A_CHARTEXT) != c; cc++);
  *ch = cc % 0x100;
  wmove (crt_ActivePanel -> w, sy, sx);
}

void crt_fillwin (Char ch, TTextAttr attr);
void crt_fillwin (Char ch, TTextAttr attr)
{
  check_winchanged ();
  wbkgdset (crt_ActivePanel -> w, attr2cursattr (attr) | chtransform (ch, attr));
  werase (crt_ActivePanel -> w);
  crt_setattr ();
  crt_need_refresh ();
}

size_t crt_write (void *PrivateData, const Char *buffer, size_t size);
size_t crt_write (void *PrivateData, const Char *buffer, size_t size)
{
  size_t i;
  Char ch;
  (void) PrivateData;
  crt_setattr ();
  for (i = 0; i < size; i++)
    {
      ch = buffer [i];
      if (crt_ActivePanel -> UseControlChars)
        {
          if (ch == chBell)
            {
              crt_beep ();
              continue;
            }
          else if (ch == chBkSp)
            {
              int x, y;
              getyx (crt_ActivePanel -> w, y, x);
              if (x > 0)
                wmove (crt_ActivePanel -> w, y, x - 1);
              continue;
            }
          else if (ch == chLF)
            {
              int x, y;
              getyx (crt_ActivePanel -> w, y, x);
              if (y + WindYMin >= WindYMax)
                {
                  if (crt_ActivePanel -> ScrollState) wscrl (crt_ActivePanel -> w, 1);
                  wmove (crt_ActivePanel -> w, y, 0);
                }
              else
                wmove (crt_ActivePanel -> w, y + 1, 0);
              continue;
            }
          else if (ch == chCR)
            {
              int x, y;
              getyx (crt_ActivePanel -> w, y, x);
              wmove (crt_ActivePanel -> w, y, 0);
              continue;
            }
        }
      waddch (crt_ActivePanel -> w, chtransform (ch, crt_TextAttr));
    }
  crt_need_refresh ();
  return size;
}

void crt_writecharattrat (int x, int y, int Count, TCharAttr *CharAttr);
void crt_writecharattrat (int x, int y, int Count, TCharAttr *CharAttr)
{
  int m = WindXMax - WindXMin + 1 - x + 1;
  if (Count > m) Count = m;
  if (Count > 0)
    {
      chtype buf [Count];
      int cattr = 0, LastAttr = - 1, sx, sy, i;
      TTextAttr Attr;
      for (i = 0; i < Count; i++)
        {
          Attr = CharAttr [i].Attr;
          if (Attr != LastAttr)
            {
              cattr = attr2cursattr (Attr);
              LastAttr = Attr;
            }
          buf [i] = chtransform (CharAttr [i].Ch, Attr) | cattr;
        }
      crt_setattr ();
      getyx (crt_ActivePanel -> w, sy, sx);
      wmove (crt_ActivePanel -> w, y - 1, x - 1);
      crt_setscroll (FALSE);
      waddchnstr (crt_ActivePanel -> w, buf, Count);
      crt_setscroll (crt_ActivePanel -> ScrollState);
      wmove (crt_ActivePanel -> w, sy, sx);
      crt_need_refresh ();
    }
}
